/**
 * CV Maker PHP - Main JavaScript Application
 * Handles frontend interactions and AJAX requests
 */

class CVMakerApp {
    constructor() {
        this.currentTab = 'personal-info';
        this.userData = {};
        this.aiSettings = {};
        this.sectionNames = {};
        this.sectionOrder = [];
        this.isGenerating = false;
        
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadFormData();
        this.initializeTabs();
        this.loadSettings();
    }

    bindEvents() {
        // Tab navigation
        document.querySelectorAll('.tab-nav button').forEach(button => {
            button.addEventListener('click', (e) => {
                this.switchTab(e.target.dataset.tab);
            });
        });

        // Form submissions
        document.getElementById('cv-form')?.addEventListener('submit', (e) => {
            e.preventDefault();
            this.generateCV();
        });

        // Settings forms
        document.getElementById('ai-settings-form')?.addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveAISettings();
        });

        // Test API connection
        document.getElementById('test-api-btn')?.addEventListener('click', () => {
            this.testAPIConnection();
        });

        // Add/Edit/Delete buttons for lists
        this.bindListEvents();

        // Template selection
        document.querySelectorAll('.template-card').forEach(card => {
            card.addEventListener('click', () => {
                this.selectTemplate(card.dataset.template);
            });
        });

        // Reset settings
        document.getElementById('reset-settings-btn')?.addEventListener('click', () => {
            this.showResetDialog();
        });

        // Auto-save form data
        this.bindAutoSave();
    }

    switchTab(tabId) {
        // Update navigation
        document.querySelectorAll('.tab-nav button').forEach(btn => {
            btn.classList.remove('active');
        });
        document.querySelector(`[data-tab="${tabId}"]`).classList.add('active');

        // Update content
        document.querySelectorAll('.tab-content').forEach(content => {
            content.classList.remove('active');
        });
        document.getElementById(tabId).classList.add('active');

        this.currentTab = tabId;
    }

    initializeTabs() {
        // Show first tab by default
        const firstTab = document.querySelector('.tab-nav button');
        if (firstTab) {
            this.switchTab(firstTab.dataset.tab);
        }
    }

    loadFormData() {
        // Load user data from session/localStorage if available
        const savedData = localStorage.getItem('cv_maker_form_data');
        if (savedData) {
            try {
                this.userData = JSON.parse(savedData);
                this.populateForm();
            } catch (e) {
                console.error('Error loading saved form data:', e);
            }
        }
    }

    populateForm() {
        // Populate basic fields
        Object.keys(this.userData).forEach(key => {
            const input = document.getElementById(key);
            if (input && typeof this.userData[key] === 'string') {
                input.value = this.userData[key];
            }
        });

        // Populate lists
        this.populateList('experiences', this.userData.experiences || []);
        this.populateList('certifications', this.userData.certifications || []);
        this.populateList('languages', this.userData.languages || []);
    }

    populateList(listType, items) {
        const container = document.getElementById(`${listType}-list`);
        if (!container) return;

        container.innerHTML = '';
        items.forEach((item, index) => {
            const listItem = this.createListItem(listType, item, index);
            container.appendChild(listItem);
        });
    }

    createListItem(listType, item, index) {
        const div = document.createElement('div');
        div.className = 'list-item';
        div.dataset.index = index;

        let content = '';
        switch (listType) {
            case 'experiences':
                content = `
                    <div class="list-item-content">
                        <div class="list-item-title">${item.position || 'Position'}</div>
                        <div class="list-item-subtitle">${item.company || 'Company'} • ${item.duration || 'Duration'}</div>
                    </div>
                `;
                break;
            case 'certifications':
                content = `
                    <div class="list-item-content">
                        <div class="list-item-title">${item.name || 'Certification'}</div>
                        <div class="list-item-subtitle">${item.authority || 'Authority'}</div>
                    </div>
                `;
                break;
            case 'languages':
                content = `
                    <div class="list-item-content">
                        <div class="list-item-title">${item.name || 'Language'}</div>
                        <div class="list-item-subtitle">${item.proficiency || 'Proficiency'}</div>
                    </div>
                `;
                break;
        }

        div.innerHTML = content + `
            <div class="list-item-actions">
                <button type="button" class="btn btn-sm btn-outline" onclick="cvApp.editListItem('${listType}', ${index})">Edit</button>
                <button type="button" class="btn btn-sm btn-danger" onclick="cvApp.deleteListItem('${listType}', ${index})">Delete</button>
            </div>
        `;

        return div;
    }

    bindListEvents() {
        // Add buttons
        document.querySelectorAll('[data-action="add"]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const listType = e.target.dataset.list;
                this.showListItemModal(listType);
            });
        });
    }

    showListItemModal(listType, index = null) {
        const modal = document.getElementById('list-item-modal');
        const form = document.getElementById('list-item-form');
        const title = document.getElementById('modal-title');
        
        // Clear form
        form.reset();
        
        // Set modal title and form fields based on list type
        const isEdit = index !== null;
        title.textContent = `${isEdit ? 'Edit' : 'Add'} ${this.getListTypeLabel(listType)}`;
        
        // Configure form fields
        this.configureModalForm(listType, isEdit ? this.getUserDataList(listType)[index] : null);
        
        // Store current context
        modal.dataset.listType = listType;
        modal.dataset.index = index;
        
        // Show modal
        modal.classList.add('active');
    }

    configureModalForm(listType, data = null) {
        const formFields = document.getElementById('modal-form-fields');
        let fieldsHTML = '';

        switch (listType) {
            case 'experiences':
                fieldsHTML = `
                    <div class="form-group">
                        <label class="form-label">Position</label>
                        <input type="text" name="position" class="form-input" value="${data?.position || ''}" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Company</label>
                        <input type="text" name="company" class="form-input" value="${data?.company || ''}" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Duration</label>
                        <input type="text" name="duration" class="form-input" value="${data?.duration || ''}" placeholder="e.g., Jan 2020 - Present" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Details (one per line)</label>
                        <textarea name="details" class="form-textarea" placeholder="• Managed team of 5 developers&#10;• Increased productivity by 30%">${data?.details ? data.details.join('\n• ') : ''}</textarea>
                    </div>
                `;
                break;
            case 'certifications':
                fieldsHTML = `
                    <div class="form-group">
                        <label class="form-label">Certification Name</label>
                        <input type="text" name="name" class="form-input" value="${data?.name || ''}" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Issuing Authority</label>
                        <input type="text" name="authority" class="form-input" value="${data?.authority || ''}" required>
                    </div>
                `;
                break;
            case 'languages':
                fieldsHTML = `
                    <div class="form-group">
                        <label class="form-label">Language</label>
                        <input type="text" name="name" class="form-input" value="${data?.name || ''}" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Proficiency Level</label>
                        <select name="proficiency" class="form-select" required>
                            <option value="">Select proficiency</option>
                            <option value="Native" ${data?.proficiency === 'Native' ? 'selected' : ''}>Native</option>
                            <option value="Fluent" ${data?.proficiency === 'Fluent' ? 'selected' : ''}>Fluent</option>
                            <option value="Advanced" ${data?.proficiency === 'Advanced' ? 'selected' : ''}>Advanced</option>
                            <option value="Intermediate" ${data?.proficiency === 'Intermediate' ? 'selected' : ''}>Intermediate</option>
                            <option value="Basic" ${data?.proficiency === 'Basic' ? 'selected' : ''}>Basic</option>
                        </select>
                    </div>
                `;
                break;
        }

        formFields.innerHTML = fieldsHTML;
    }

    saveListItem() {
        const modal = document.getElementById('list-item-modal');
        const form = document.getElementById('list-item-form');
        const listType = modal.dataset.listType;
        const index = modal.dataset.index;
        
        if (!form.checkValidity()) {
            form.reportValidity();
            return;
        }

        const formData = new FormData(form);
        const itemData = {};
        
        for (let [key, value] of formData.entries()) {
            if (key === 'details' && value) {
                // Split details by lines and clean up
                itemData[key] = value.split('\n')
                    .map(line => line.replace(/^[•\-\*]\s*/, '').trim())
                    .filter(line => line.length > 0);
            } else {
                itemData[key] = value;
            }
        }

        // Update data
        if (!this.userData[listType]) {
            this.userData[listType] = [];
        }

        if (index !== null && index !== 'null') {
            this.userData[listType][parseInt(index)] = itemData;
        } else {
            this.userData[listType].push(itemData);
        }

        // Update UI
        this.populateList(listType, this.userData[listType]);
        
        // Save to localStorage
        this.saveFormData();
        
        // Close modal
        this.closeModal();
        
        this.showAlert('success', `${this.getListTypeLabel(listType)} saved successfully!`);
    }

    editListItem(listType, index) {
        this.showListItemModal(listType, index);
    }

    deleteListItem(listType, index) {
        if (confirm(`Are you sure you want to delete this ${this.getListTypeLabel(listType).toLowerCase()}?`)) {
            this.userData[listType].splice(index, 1);
            this.populateList(listType, this.userData[listType]);
            this.saveFormData();
            this.showAlert('success', `${this.getListTypeLabel(listType)} deleted successfully!`);
        }
    }

    getUserDataList(listType) {
        return this.userData[listType] || [];
    }

    getListTypeLabel(listType) {
        const labels = {
            'experiences': 'Experience',
            'certifications': 'Certification',
            'languages': 'Language'
        };
        return labels[listType] || listType;
    }

    closeModal() {
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.classList.remove('active');
        });
    }

    bindAutoSave() {
        // Auto-save form data on input changes
        document.querySelectorAll('input, textarea, select').forEach(input => {
            if (input.form && input.form.id === 'cv-form') {
                input.addEventListener('input', () => {
                    this.collectFormData();
                    this.saveFormData();
                });
            }
        });
    }

    collectFormData() {
        const form = document.getElementById('cv-form');
        if (!form) return;

        const formData = new FormData(form);
        for (let [key, value] of formData.entries()) {
            if (!['certifications', 'languages', 'experiences'].includes(key)) {
                this.userData[key] = value;
            }
        }
    }

    saveFormData() {
        localStorage.setItem('cv_maker_form_data', JSON.stringify(this.userData));
    }

    async generateCV() {
        if (this.isGenerating) return;

        this.isGenerating = true;
        this.updateGenerateButton(true);

        try {
            // Collect all form data
            this.collectFormData();

            const formData = new FormData();
            
            // Add user data
            Object.keys(this.userData).forEach(key => {
                if (Array.isArray(this.userData[key])) {
                    formData.append(key, JSON.stringify(this.userData[key]));
                } else {
                    formData.append(key, this.userData[key] || '');
                }
            });

            // Add job description
            const jobDescription = document.getElementById('job_description')?.value || '';
            formData.append('job_description', jobDescription);

            // Add template
            const selectedTemplate = document.querySelector('.template-card.selected')?.dataset.template || 'modern';
            formData.append('template', selectedTemplate);

            const response = await fetch('/generate-cv', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.showAlert('success', result.message);
                this.showDownloadLink(result.download_url, result.file_name);
            } else {
                this.showAlert('error', result.error || 'Failed to generate CV');
            }

        } catch (error) {
            console.error('Error generating CV:', error);
            this.showAlert('error', 'An error occurred while generating the CV. Please try again.');
        } finally {
            this.isGenerating = false;
            this.updateGenerateButton(false);
        }
    }

    updateGenerateButton(isLoading) {
        const button = document.getElementById('generate-cv-btn');
        if (!button) return;

        if (isLoading) {
            button.disabled = true;
            button.innerHTML = '<span class="spinner"></span> Generating CV...';
        } else {
            button.disabled = false;
            button.innerHTML = 'Generate CV';
        }
    }

    showDownloadLink(downloadUrl, fileName) {
        const alertDiv = document.createElement('div');
        alertDiv.className = 'alert alert-success';
        alertDiv.innerHTML = `
            <span>CV generated successfully!</span>
            <a href="${downloadUrl}" download="${fileName}" class="btn btn-sm btn-success" style="margin-left: 1rem;">
                Download CV
            </a>
        `;

        // Insert after the generate button
        const generateBtn = document.getElementById('generate-cv-btn');
        generateBtn.parentNode.insertBefore(alertDiv, generateBtn.nextSibling);

        // Remove after 10 seconds
        setTimeout(() => {
            alertDiv.remove();
        }, 10000);
    }

    async saveAISettings() {
        try {
            const form = document.getElementById('ai-settings-form');
            const formData = new FormData(form);
            
            const settings = {};
            for (let [key, value] of formData.entries()) {
                if (key === 'excluded_terms') {
                    settings[key] = value.split(',').map(term => term.trim()).filter(term => term);
                } else {
                    settings[key] = value;
                }
            }

            const response = await fetch('/save-settings', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `setting_type=ai_settings&data=${encodeURIComponent(JSON.stringify(settings))}`
            });

            const result = await response.json();

            if (result.success) {
                this.aiSettings = settings;
                this.showAlert('success', 'AI settings saved successfully!');
            } else {
                this.showAlert('error', result.error || 'Failed to save settings');
            }

        } catch (error) {
            console.error('Error saving AI settings:', error);
            this.showAlert('error', 'An error occurred while saving settings');
        }
    }

    async testAPIConnection() {
        const button = document.getElementById('test-api-btn');
        const originalText = button.textContent;
        
        button.disabled = true;
        button.innerHTML = '<span class="spinner"></span> Testing...';

        try {
            const form = document.getElementById('ai-settings-form');
            const formData = new FormData(form);

            const response = await fetch('/test-api', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.showAlert('success', 'API connection successful!');
            } else {
                this.showAlert('error', result.message || 'API connection failed');
            }

        } catch (error) {
            console.error('Error testing API:', error);
            this.showAlert('error', 'Failed to test API connection');
        } finally {
            button.disabled = false;
            button.textContent = originalText;
        }
    }

    selectTemplate(templateName) {
        document.querySelectorAll('.template-card').forEach(card => {
            card.classList.remove('selected');
        });
        
        document.querySelector(`[data-template="${templateName}"]`).classList.add('selected');
        
        // Save template preference
        this.saveTemplateSetting(templateName);
    }

    async saveTemplateSetting(template) {
        try {
            const response = await fetch('/save-settings', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `setting_type=template_preference&data=${encodeURIComponent(JSON.stringify(template))}`
            });

            const result = await response.json();
            if (!result.success) {
                console.error('Failed to save template preference:', result.error);
            }

        } catch (error) {
            console.error('Error saving template preference:', error);
        }
    }

    loadSettings() {
        // Load settings from server-side rendered data or make AJAX call
        // This would typically be populated from PHP template
    }

    showResetDialog() {
        // Implementation for reset dialog
        if (confirm('Are you sure you want to reset all settings? This action cannot be undone.')) {
            this.resetSettings(['personal_info', 'experiences', 'certifications', 'languages', 'ai_settings']);
        }
    }

    async resetSettings(settingsToReset) {
        try {
            const response = await fetch('/reset-settings', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `settings=${encodeURIComponent(JSON.stringify(settingsToReset))}`
            });

            const result = await response.json();

            if (result.success) {
                this.showAlert('success', 'Settings reset successfully!');
                // Reload page to reflect changes
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            } else {
                this.showAlert('error', result.error || 'Failed to reset settings');
            }

        } catch (error) {
            console.error('Error resetting settings:', error);
            this.showAlert('error', 'An error occurred while resetting settings');
        }
    }

    showAlert(type, message) {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;

        // Insert at top of main content
        const mainContent = document.querySelector('.main-content');
        mainContent.insertBefore(alertDiv, mainContent.firstChild);

        // Auto-remove after 5 seconds
        setTimeout(() => {
            alertDiv.remove();
        }, 5000);

        // Scroll to top to show alert
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }
}

// Initialize app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.cvApp = new CVMakerApp();
});

// Modal event handlers
document.addEventListener('click', (e) => {
    if (e.target.classList.contains('modal-overlay')) {
        cvApp.closeModal();
    }
    
    if (e.target.classList.contains('modal-close')) {
        cvApp.closeModal();
    }
});

// Save list item handler
function saveListItem() {
    cvApp.saveListItem();
}
