<?php

namespace CVMaker\Controllers;

use CVMaker\Services\OpenAIService;
use CVMaker\Services\DocumentGenerator;
use CVMaker\Services\SettingsManager;

/**
 * CV Controller - Main controller for CV generation
 * Handles web requests and coordinates services
 */
class CVController
{
    private $settingsManager;
    private $config;

    public function __construct()
    {
        $this->config = include __DIR__ . '/../../config/config.php';
        $this->settingsManager = new SettingsManager();
        
        // Start session if not already started
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }

    /**
     * Display the main CV maker form
     */
    public function index()
    {
        // Load user data and settings
        $userData = $this->settingsManager->loadUserData();
        $aiSettings = $this->settingsManager->loadAISettings();
        $sectionNames = $this->settingsManager->loadSectionNames();
        $sectionOrder = $this->settingsManager->loadSectionOrder();
        $templatePreference = $this->settingsManager->loadTemplatePreference();

        // Store in session for form persistence
        $_SESSION['user_data'] = $userData;
        $_SESSION['ai_settings'] = $aiSettings;
        $_SESSION['section_names'] = $sectionNames;
        $_SESSION['section_order'] = $sectionOrder;
        $_SESSION['template_preference'] = $templatePreference;

        // Load the main template
        $this->loadTemplate('home', [
            'userData' => $userData,
            'aiSettings' => $aiSettings,
            'sectionNames' => $sectionNames,
            'sectionOrder' => $sectionOrder,
            'templatePreference' => $templatePreference,
            'config' => $this->config
        ]);
    }

    /**
     * Generate CV document
     */
    public function generateCV()
    {
        try {
            // Validate request method
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                return $this->jsonResponse(['error' => 'Invalid request method'], 405);
            }

            // Get form data
            $userData = $this->getFormData();
            $aiSettings = $_SESSION['ai_settings'] ?? [];
            $sectionNames = $_SESSION['section_names'] ?? [];
            $sectionOrder = $_SESSION['section_order'] ?? [];
            $template = $_POST['template'] ?? 'modern';

            // Validate required fields
            $validation = $this->validateCVData($userData, $aiSettings);
            if (!$validation['valid']) {
                return $this->jsonResponse(['error' => $validation['message']], 400);
            }

            // Save user data
            $this->settingsManager->saveUserData($userData);

            // Initialize OpenAI service
            $openAIService = new OpenAIService(
                $aiSettings['api_key'],
                $aiSettings['model'],
                $aiSettings['provider'],
                $this->getAPIHeaders($aiSettings)
            );

            // Generate AI content
            $result = $openAIService->generateCVContent(
                $_POST['job_description'] ?? '',
                $aiSettings['excluded_terms'] ?? [],
                $sectionNames
            );

            if (isset($result['error'])) {
                return $this->jsonResponse(['error' => $result['error']], 500);
            }

            // Generate document
            $documentGenerator = new DocumentGenerator();
            $phpWord = $documentGenerator->generateCV(
                $userData,
                $result['data'],
                $sectionNames,
                $sectionOrder,
                $template
            );

            // Save document
            $fileName = 'CV_' . preg_replace('/[^a-zA-Z0-9_-]/', '_', $userData['name']) . '_' . date('Y-m-d_H-i-s') . '.docx';
            $filePath = $this->config['paths']['downloads'] . '/' . $fileName;

            $saveResult = $documentGenerator->saveDocument($filePath);
            
            if (!$saveResult['success']) {
                return $this->jsonResponse(['error' => $saveResult['message']], 500);
            }

            // Return success response with download link
            return $this->jsonResponse([
                'success' => true,
                'message' => 'CV generated successfully',
                'download_url' => '/downloads/' . $fileName,
                'file_name' => $fileName
            ]);

        } catch (\Exception $e) {
            return $this->jsonResponse(['error' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Save settings via AJAX
     */
    public function saveSettings()
    {
        try {
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                return $this->jsonResponse(['error' => 'Invalid request method'], 405);
            }

            $settingType = $_POST['setting_type'] ?? '';
            $data = json_decode($_POST['data'] ?? '{}', true);

            switch ($settingType) {
                case 'ai_settings':
                    $this->settingsManager->saveAISettings($data);
                    $_SESSION['ai_settings'] = $data;
                    break;
                case 'section_names':
                    $this->settingsManager->saveSectionNames($data);
                    $_SESSION['section_names'] = $data;
                    break;
                case 'section_order':
                    $this->settingsManager->saveSectionOrder($data);
                    $_SESSION['section_order'] = $data;
                    break;
                case 'template_preference':
                    $this->settingsManager->saveTemplatePreference($data);
                    $_SESSION['template_preference'] = $data;
                    break;
                default:
                    return $this->jsonResponse(['error' => 'Invalid setting type'], 400);
            }

            return $this->jsonResponse(['success' => true, 'message' => 'Settings saved successfully']);

        } catch (\Exception $e) {
            return $this->jsonResponse(['error' => 'Failed to save settings: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Test API connection
     */
    public function testAPI()
    {
        try {
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                return $this->jsonResponse(['error' => 'Invalid request method'], 405);
            }

            $apiKey = $_POST['api_key'] ?? '';
            $model = $_POST['model'] ?? 'gpt-3.5-turbo';
            $provider = $_POST['provider'] ?? 'openai';

            if (empty($apiKey)) {
                return $this->jsonResponse(['error' => 'API key is required'], 400);
            }

            $headers = $this->getAPIHeaders($_POST);
            $openAIService = new OpenAIService($apiKey, $model, $provider, $headers);
            
            $result = $openAIService->testConnection();
            
            return $this->jsonResponse($result);

        } catch (\Exception $e) {
            return $this->jsonResponse(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Reset settings
     */
    public function resetSettings()
    {
        try {
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                return $this->jsonResponse(['error' => 'Invalid request method'], 405);
            }

            $settingsToReset = json_decode($_POST['settings'] ?? '[]', true);
            
            if (empty($settingsToReset)) {
                return $this->jsonResponse(['error' => 'No settings specified for reset'], 400);
            }

            $result = $this->settingsManager->resetSettings($settingsToReset);
            
            // Clear session data for reset settings
            foreach ($settingsToReset as $setting) {
                switch ($setting) {
                    case 'personal_info':
                    case 'experiences':
                    case 'certifications':
                    case 'languages':
                        $_SESSION['user_data'] = $this->settingsManager->loadUserData();
                        break;
                    case 'ai_settings':
                        $_SESSION['ai_settings'] = $this->settingsManager->loadAISettings();
                        break;
                    case 'section_names':
                        $_SESSION['section_names'] = $this->settingsManager->loadSectionNames();
                        break;
                    case 'section_order':
                        $_SESSION['section_order'] = $this->settingsManager->loadSectionOrder();
                        break;
                    case 'template_preference':
                        $_SESSION['template_preference'] = $this->settingsManager->loadTemplatePreference();
                        break;
                }
            }

            return $this->jsonResponse($result);

        } catch (\Exception $e) {
            return $this->jsonResponse(['error' => 'Failed to reset settings: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get form data from POST request
     */
    private function getFormData()
    {
        return [
            'name' => $_POST['name'] ?? '',
            'title' => $_POST['title'] ?? '',
            'email' => $_POST['email'] ?? '',
            'linkedin' => $_POST['linkedin'] ?? '',
            'phone' => $_POST['phone'] ?? '',
            'location' => $_POST['location'] ?? '',
            'university' => $_POST['university'] ?? '',
            'degree' => $_POST['degree'] ?? '',
            'certifications' => json_decode($_POST['certifications'] ?? '[]', true),
            'languages' => json_decode($_POST['languages'] ?? '[]', true),
            'experiences' => json_decode($_POST['experiences'] ?? '[]', true)
        ];
    }

    /**
     * Validate CV generation data
     */
    private function validateCVData($userData, $aiSettings)
    {
        if (empty($userData['name'])) {
            return ['valid' => false, 'message' => 'Name is required'];
        }

        if (empty($aiSettings['api_key'])) {
            return ['valid' => false, 'message' => 'API key is required'];
        }

        if (empty($aiSettings['model'])) {
            return ['valid' => false, 'message' => 'AI model is required'];
        }

        if (empty($_POST['job_description'])) {
            return ['valid' => false, 'message' => 'Job description is required'];
        }

        return ['valid' => true];
    }

    /**
     * Get API headers based on provider
     */
    private function getAPIHeaders($settings)
    {
        $headers = [];
        
        if (($settings['provider'] ?? '') === 'openrouter') {
            if (!empty($settings['http_referer'])) {
                $headers['HTTP-Referer'] = $settings['http_referer'];
            }
            if (!empty($settings['x_title'])) {
                $headers['X-Title'] = $settings['x_title'];
            }
        }
        
        return $headers;
    }

    /**
     * Load template file
     */
    private function loadTemplate($template, $data = [])
    {
        extract($data);
        $templatePath = $this->config['paths']['templates'] . '/' . $template . '.php';
        
        if (file_exists($templatePath)) {
            include $templatePath;
        } else {
            http_response_code(404);
            echo "Template not found: " . $template;
        }
    }

    /**
     * Return JSON response
     */
    private function jsonResponse($data, $statusCode = 200)
    {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }
}
