<?php

namespace CVMaker\Services;

use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\Style\Font;
use PhpOffice\PhpWord\Style\Paragraph;
use PhpOffice\PhpWord\Shared\Converter;
use PhpOffice\PhpWord\Element\Section;

/**
 * Document Generator Service - Handles Word document creation
 * Converted from Python's doc_generator.py
 */
class DocumentGenerator
{
    private $phpWord;
    private $section;

    public function __construct()
    {
        $this->phpWord = new PhpWord();
        
        // Set document properties
        $properties = $this->phpWord->getDocInfo();
        $properties->setCreator('CV Maker PHP');
        $properties->setCompany('CV Maker');
        $properties->setTitle('Professional CV');
        $properties->setDescription('Generated CV using CV Maker PHP Application');
        
        // Set default font
        $this->phpWord->setDefaultFontName('Calibri');
        $this->phpWord->setDefaultFontSize(10.5);
    }

    /**
     * Helper function to set font properties for a text run
     * Equivalent to set_run_font() in Python
     */
    private function getFontStyle($name = 'Calibri', $size = 10.5, $bold = false, $italic = false, $color = null)
    {
        $fontStyle = [];
        $fontStyle['name'] = $name;
        $fontStyle['size'] = $size;
        $fontStyle['bold'] = $bold;
        $fontStyle['italic'] = $italic;
        
        if ($color) {
            $fontStyle['color'] = $color;
        }
        
        return $fontStyle;
    }

    /**
     * Add a section title with decorative lines
     * Equivalent to add_new_section_title() in Python
     */
    private function addSectionTitle($title)
    {
        // Add some space before section if not first element
        if (count($this->section->getElements()) > 0) {
            $this->section->addTextBreak();
        }

        // Create a table with 3 columns for the title design
        $table = $this->section->addTable([
            'alignment' => \PhpOffice\PhpWord\SimpleType\JcTable::CENTER,
            'cellMargin' => 0
        ]);
        
        $table->addRow();
        
        // Left cell with line
        $leftCell = $table->addCell(Converter::inchToTwip(2.5), [
            'valign' => 'center'
        ]);
        $leftCell->addText('__________________________________', 
            $this->getFontStyle('Calibri', 10.5), 
            ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::RIGHT]
        );
        
        // Center cell with title
        $centerCell = $table->addCell(Converter::inchToTwip(2.5), [
            'valign' => 'center'
        ]);
        $centerCell->addText(strtoupper($title), 
            $this->getFontStyle('Calibri', 12, true), 
            ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::CENTER]
        );
        
        // Right cell with line
        $rightCell = $table->addCell(Converter::inchToTwip(2.5), [
            'valign' => 'center'
        ]);
        $rightCell->addText('__________________________________', 
            $this->getFontStyle('Calibri', 10.5), 
            ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::LEFT]
        );

        // Add space after section title
        $this->section->addTextBreak();
    }

    /**
     * Generate CV document
     * Equivalent to write_cv() in Python
     */
    public function generateCV($userData, $aiData, $sectionNames, $sectionOrder, $template = 'modern')
    {
        // Create section with margins
        $this->section = $this->phpWord->addSection([
            'marginTop' => Converter::inchToTwip(0.7),
            'marginBottom' => Converter::inchToTwip(0.7),
            'marginLeft' => Converter::inchToTwip(0.7),
            'marginRight' => Converter::inchToTwip(0.7)
        ]);

        // Header section with name and contact info
        $this->addHeader($userData);

        // Process sections in the specified order
        foreach ($sectionOrder as $sectionKey) {
            $this->addSection($sectionKey, $userData, $aiData, $sectionNames);
        }

        return $this->phpWord;
    }

    /**
     * Add header section with name and contact information
     */
    private function addHeader($userData)
    {
        // Name
        if (!empty($userData['name'])) {
            $this->section->addText($userData['name'], 
                $this->getFontStyle('Calibri', 26, true), 
                ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::CENTER, 'spaceAfter' => 80]
            );
        }

        // Title/Position
        if (!empty($userData['title'])) {
            $this->section->addText($userData['title'], 
                $this->getFontStyle('Calibri', 14), 
                ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::CENTER, 'spaceAfter' => 120]
            );
        }

        // Contact information
        $contactInfo = [];
        if (!empty($userData['location'])) $contactInfo[] = $userData['location'];
        if (!empty($userData['phone'])) $contactInfo[] = $userData['phone'];
        if (!empty($userData['email'])) $contactInfo[] = $userData['email'];
        if (!empty($userData['linkedin'])) $contactInfo[] = $userData['linkedin'];

        if (!empty($contactInfo)) {
            $this->section->addText(implode('  ♦  ', $contactInfo), 
                $this->getFontStyle('Calibri', 10.5), 
                ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::CENTER, 'spaceAfter' => 120]
            );
        }

        // Header line
        $this->section->addText('_________________________________________________________________', 
            $this->getFontStyle('Calibri', 10.5), 
            ['alignment' => \PhpOffice\PhpWord\SimpleType\Jc::CENTER, 'spaceAfter' => 160]
        );
    }

    /**
     * Add individual sections based on type
     */
    private function addSection($sectionKey, $userData, $aiData, $sectionNames)
    {
        switch ($sectionKey) {
            case 'profile':
                $this->addProfileSection($aiData, $sectionNames);
                break;
            case 'experiences':
                $this->addExperienceSection($userData, $sectionNames);
                break;
            case 'education':
                $this->addEducationSection($userData, $sectionNames);
                break;
            case 'certifications':
                $this->addCertificationsSection($userData, $sectionNames);
                break;
            case 'languages':
                $this->addLanguagesSection($userData, $sectionNames);
                break;
            default:
                $this->addDynamicSection($sectionKey, $aiData, $sectionNames);
                break;
        }
    }

    /**
     * Add profile/summary section
     */
    private function addProfileSection($aiData, $sectionNames)
    {
        if (!empty($aiData['profile'])) {
            $sectionName = $sectionNames['profile'] ?? 'Professional Summary';
            $this->addSectionTitle($sectionName);
            $this->section->addText($aiData['profile'], 
                $this->getFontStyle('Calibri', 10.5), 
                ['spaceAfter' => 80]
            );
        }
    }

    /**
     * Add experience section
     */
    private function addExperienceSection($userData, $sectionNames)
    {
        if (!empty($userData['experiences'])) {
            $sectionName = $sectionNames['experiences'] ?? 'Experience';
            $this->addSectionTitle($sectionName);

            foreach ($userData['experiences'] as $exp) {
                // Job title and duration
                $titleText = $this->section->addTextRun(['spaceAfter' => 0]);
                $titleText->addText($exp['position'] ?? '', $this->getFontStyle('Calibri', 10.5, true));
                $titleText->addTab();
                $titleText->addText($exp['duration'] ?? '', $this->getFontStyle('Calibri', 10.5, false, true));

                // Company
                $this->section->addText($exp['company'] ?? '', 
                    $this->getFontStyle('Calibri', 10.5, false, true), 
                    ['spaceAfter' => 80]
                );

                // Details as bullet points
                if (!empty($exp['details'])) {
                    foreach ($exp['details'] as $detail) {
                        $this->section->addListItem($detail, 0, 
                            $this->getFontStyle('Calibri', 10.5), 
                            null, 
                            ['spaceAfter' => 40]
                        );
                    }
                }

                $this->section->addTextBreak();
            }
        }
    }

    /**
     * Add education section
     */
    private function addEducationSection($userData, $sectionNames)
    {
        if (!empty($userData['degree'])) {
            $sectionName = $sectionNames['education'] ?? 'Education';
            $this->addSectionTitle($sectionName);

            $this->section->addText($userData['degree'], 
                $this->getFontStyle('Calibri', 10.5, true), 
                ['spaceAfter' => 0]
            );

            if (!empty($userData['university'])) {
                $this->section->addText($userData['university'], 
                    $this->getFontStyle('Calibri', 10.5), 
                    ['spaceAfter' => 160]
                );
            }
        }
    }

    /**
     * Add certifications section
     */
    private function addCertificationsSection($userData, $sectionNames)
    {
        if (!empty($userData['certifications'])) {
            $sectionName = $sectionNames['certifications'] ?? 'Certifications';
            $this->addSectionTitle($sectionName);

            foreach ($userData['certifications'] as $cert) {
                $certText = $this->section->addTextRun(['spaceAfter' => 80]);
                $certText->addText($cert['name'] ?? '', $this->getFontStyle('Calibri', 10.5, true));
                if (!empty($cert['authority'])) {
                    $certText->addText(' - ' . $cert['authority'], $this->getFontStyle('Calibri', 10.5));
                }
            }
        }
    }

    /**
     * Add languages section
     */
    private function addLanguagesSection($userData, $sectionNames)
    {
        if (!empty($userData['languages'])) {
            $sectionName = $sectionNames['languages'] ?? 'Languages';
            $this->addSectionTitle($sectionName);

            foreach ($userData['languages'] as $lang) {
                $langText = $this->section->addTextRun(['spaceAfter' => 40]);
                $langText->addText(($lang['name'] ?? '') . ': ', $this->getFontStyle('Calibri', 10.5, true));
                $langText->addText($lang['proficiency'] ?? '', $this->getFontStyle('Calibri', 10.5));
            }
        }
    }

    /**
     * Add dynamic sections (skills, interests, etc.)
     */
    private function addDynamicSection($sectionKey, $aiData, $sectionNames)
    {
        if (!empty($aiData[$sectionKey]) && !empty($sectionNames[$sectionKey])) {
            $this->addSectionTitle($sectionNames[$sectionKey]);
            
            $items = $aiData[$sectionKey];
            
            // Special handling for skills - use two columns
            if ($sectionKey === 'skills' && !empty($items)) {
                $numRows = ceil(count($items) / 2);
                $table = $this->section->addTable();
                
                for ($i = 0; $i < $numRows; $i++) {
                    $table->addRow();
                    
                    // First column
                    if (isset($items[$i * 2])) {
                        $cell1 = $table->addCell(Converter::inchToTwip(3.75));
                        $cell1->addText('•  ' . $items[$i * 2], $this->getFontStyle('Calibri', 10.5));
                    }
                    
                    // Second column
                    if (isset($items[$i * 2 + 1])) {
                        $cell2 = $table->addCell(Converter::inchToTwip(3.75));
                        $cell2->addText('•  ' . $items[$i * 2 + 1], $this->getFontStyle('Calibri', 10.5));
                    }
                }
            } else {
                // Regular bullet list
                foreach ($items as $item) {
                    $this->section->addListItem($item, 0, 
                        $this->getFontStyle('Calibri', 10.5), 
                        null, 
                        ['spaceAfter' => 40]
                    );
                }
            }
        }
    }

    /**
     * Save document to file
     */
    public function saveDocument($filePath, $format = 'Word2007')
    {
        try {
            $writer = IOFactory::createWriter($this->phpWord, $format);
            $writer->save($filePath);
            return ['success' => true, 'message' => 'Document saved successfully'];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Error saving document: ' . $e->getMessage()];
        }
    }
}
