<?php

namespace CVMaker\Services;

use Orhanerday\OpenAi\OpenAi;

/**
 * OpenAI Service - Handles AI API integration
 * Converted from Python's api_handler.py
 */
class OpenAIService
{
    private $client;
    private $model;
    private $apiKey;
    private $baseUrl;
    private $headers;

    public function __construct($apiKey, $model = 'gpt-3.5-turbo', $provider = 'openai', $headers = [])
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
        $this->headers = $headers;
        
        // Set base URL based on provider
        if ($provider === 'openrouter') {
            $this->baseUrl = 'https://openrouter.ai/api/v1';
        } else {
            $this->baseUrl = 'https://api.openai.com/v1';
        }

        // Initialize OpenAI client
        $this->client = new OpenAi($this->apiKey);
        $this->client->setBaseURL($this->baseUrl);
        
        // Set custom headers if provided
        if (!empty($this->headers)) {
            foreach ($this->headers as $key => $value) {
                $this->client->setHeader($key, $value);
            }
        }
    }

    /**
     * Analyze job description and generate CV content
     * Equivalent to analyze_job_description() in Python
     */
    public function analyzeJobDescription($jobDescription, $excludedTerms = [], $sectionNames = [])
    {
        try {
            $instructions = [];
            $formatExamples = [];
            
            $i = 1;
            foreach ($sectionNames as $key => $name) {
                if ($key === 'profile') {
                    $instructions[] = "{$i}. **Generate a {$name}:** Write a 2-3 sentence professional summary for the top of the CV, tailored to this job.";
                    $formatExamples[] = "### {$name}\nA results-oriented professional with experience in...";
                } else {
                    $instructions[] = "{$i}. **Generate {$name}:** Create a bulleted list of essential items for this section (like skills, interests, etc.) based on the job description.";
                    $formatExamples[] = "### {$name}\n- Item A\n- Item B";
                }
                $i++;
            }

            $dynamicInstructions = implode("\n", $instructions);
            $dynamicFormatExamples = implode("\n\n", $formatExamples);
            $excludedTermsStr = !empty($excludedTerms) ? implode(', ', $excludedTerms) : 'None';

            $prompt = "You are an expert CV and resume assistant. Your task is to analyze the provided job description and generate relevant, concise, and ATS-friendly content for a CV.

**Job Description:**
---
{$jobDescription}
---

**Instructions:**
{$dynamicInstructions}
- **DO NOT use the following words in your response:** {$excludedTermsStr}
- **Format your entire response clearly** with headings for each section using \"###\" followed by the exact section name provided in the instructions. Do not add any extra text before or after the content.

**Example Output Format:**
{$dynamicFormatExamples}";

            $response = $this->client->chat([
                'model' => $this->model,
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.5,
                'max_tokens' => 2000
            ]);

            $responseData = json_decode($response, true);
            
            if (isset($responseData['error'])) {
                return "Error: " . $responseData['error']['message'];
            }

            if (isset($responseData['choices'][0]['message']['content'])) {
                return $responseData['choices'][0]['message']['content'];
            }

            return "Error: Unexpected response format";

        } catch (\Exception $e) {
            return "Error: " . $e->getMessage();
        }
    }

    /**
     * Parse AI response and extract sections
     * Equivalent to parse_ai_response() in Python
     */
    public function parseAIResponse($responseText, $sectionNames)
    {
        $data = [];
        
        // Initialize data structure
        foreach ($sectionNames as $key => $name) {
            if ($key === 'profile') {
                $data[$key] = "";
            } else {
                $data[$key] = [];
            }
        }

        // Create mapping from title to key
        $titleToKeyMap = [];
        foreach ($sectionNames as $key => $name) {
            $titleToKeyMap[strtolower("### {$name}")] = $key;
        }

        $currentSectionKey = null;
        $lines = explode("\n", $responseText);

        foreach ($lines as $line) {
            $lineStripped = trim($line);
            if (empty($lineStripped)) {
                continue;
            }

            $lineLower = strtolower($lineStripped);

            // Check if line matches any section title
            $matchedTitle = null;
            foreach ($titleToKeyMap as $title => $key) {
                if (strpos($lineLower, $title) === 0) {
                    $matchedTitle = $title;
                    break;
                }
            }

            if ($matchedTitle) {
                $currentSectionKey = $titleToKeyMap[$matchedTitle];
                continue;
            }

            if ($currentSectionKey) {
                if ($currentSectionKey === 'profile') {
                    if (strpos($lineStripped, '###') !== 0) {
                        $data[$currentSectionKey] .= $lineStripped . " ";
                    }
                } elseif (strpos($lineStripped, '-') === 0 || strpos($lineStripped, '*') === 0) {
                    $content = trim(substr($lineStripped, 1));
                    if (isset($data[$currentSectionKey]) && is_array($data[$currentSectionKey])) {
                        $data[$currentSectionKey][] = $content;
                    }
                }
            }
        }

        // Clean up profile section
        if (isset($data['profile'])) {
            $data['profile'] = trim($data['profile']);
        }

        return $data;
    }

    /**
     * Complete workflow: analyze job description and return parsed data
     */
    public function generateCVContent($jobDescription, $excludedTerms = [], $sectionNames = [])
    {
        $aiResult = $this->analyzeJobDescription($jobDescription, $excludedTerms, $sectionNames);
        
        if (strpos($aiResult, "Error") === 0) {
            return ['error' => $aiResult];
        }

        $parsedData = $this->parseAIResponse($aiResult, $sectionNames);
        
        return ['success' => true, 'data' => $parsedData];
    }

    /**
     * Test API connection
     */
    public function testConnection()
    {
        try {
            $response = $this->client->listModels();
            $responseData = json_decode($response, true);
            
            if (isset($responseData['error'])) {
                return ['success' => false, 'message' => $responseData['error']['message']];
            }

            return ['success' => true, 'message' => 'API connection successful'];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}
