<?php

namespace CVMaker\Services;

/**
 * Settings Manager Service - Handles settings persistence
 * Converted from Python's settings_manager.py
 */
class SettingsManager
{
    private $settingsPath;
    private $settings;

    public function __construct($settingsPath = null)
    {
        $config = include __DIR__ . '/../../config/config.php';
        $this->settingsPath = $settingsPath ?? $config['paths']['settings'];
        
        // Ensure settings directory exists
        if (!is_dir($this->settingsPath)) {
            mkdir($this->settingsPath, 0755, true);
        }
        
        $this->settings = [];
    }

    /**
     * Load settings from file
     */
    public function loadSettings($key, $default = null)
    {
        $filePath = $this->settingsPath . '/' . $key . '.json';
        
        if (file_exists($filePath)) {
            $content = file_get_contents($filePath);
            $decoded = json_decode($content, true);
            
            if ($decoded !== null) {
                $this->settings[$key] = $decoded;
                return $decoded;
            }
        }
        
        $this->settings[$key] = $default;
        return $default;
    }

    /**
     * Save settings to file
     */
    public function saveSettings($key, $value)
    {
        try {
            $filePath = $this->settingsPath . '/' . $key . '.json';
            $content = json_encode($value, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            
            if (file_put_contents($filePath, $content) !== false) {
                $this->settings[$key] = $value;
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get setting value from memory or load from file
     */
    public function getSetting($key, $default = null)
    {
        if (isset($this->settings[$key])) {
            return $this->settings[$key];
        }
        
        return $this->loadSettings($key, $default);
    }

    /**
     * Set setting value in memory
     */
    public function setSetting($key, $value)
    {
        $this->settings[$key] = $value;
    }

    /**
     * Delete setting file
     */
    public function deleteSetting($key)
    {
        $filePath = $this->settingsPath . '/' . $key . '.json';
        
        if (file_exists($filePath)) {
            unset($this->settings[$key]);
            return unlink($filePath);
        }
        
        return true;
    }

    /**
     * Get all settings
     */
    public function getAllSettings()
    {
        return $this->settings;
    }

    /**
     * Clear all settings from memory
     */
    public function clearSettings()
    {
        $this->settings = [];
    }

    /**
     * Load user data (personal info, experiences, etc.)
     */
    public function loadUserData()
    {
        $userData = [
            'name' => '',
            'title' => '',
            'email' => '',
            'linkedin' => '',
            'phone' => '',
            'location' => '',
            'university' => '',
            'degree' => '',
            'certifications' => [],
            'languages' => [],
            'experiences' => []
        ];

        // Load personal info
        $personalInfo = $this->loadSettings('personal_info', []);
        if (!empty($personalInfo)) {
            $userData = array_merge($userData, $personalInfo);
        }

        // Load lists
        $userData['certifications'] = $this->loadSettings('certifications', []);
        $userData['languages'] = $this->loadSettings('languages', []);
        $userData['experiences'] = $this->loadSettings('experiences', []);

        return $userData;
    }

    /**
     * Save user data
     */
    public function saveUserData($userData)
    {
        // Extract personal info (non-array fields)
        $personalInfo = [];
        $lists = ['certifications', 'languages', 'experiences'];
        
        foreach ($userData as $key => $value) {
            if (!in_array($key, $lists)) {
                $personalInfo[$key] = $value;
            }
        }

        // Save personal info
        $this->saveSettings('personal_info', $personalInfo);

        // Save lists separately
        foreach ($lists as $listKey) {
            if (isset($userData[$listKey])) {
                $this->saveSettings($listKey, $userData[$listKey]);
            }
        }

        return true;
    }

    /**
     * Load AI settings
     */
    public function loadAISettings()
    {
        $defaultSettings = [
            'provider' => 'openai',
            'api_key' => '',
            'model' => 'gpt-3.5-turbo',
            'excluded_terms' => [],
            'http_referer' => '',
            'x_title' => ''
        ];

        return $this->loadSettings('ai_settings', $defaultSettings);
    }

    /**
     * Save AI settings
     */
    public function saveAISettings($settings)
    {
        return $this->saveSettings('ai_settings', $settings);
    }

    /**
     * Load section names
     */
    public function loadSectionNames()
    {
        $config = include __DIR__ . '/../../config/config.php';
        $defaultSectionNames = $config['cv_defaults']['section_names'];
        
        return $this->loadSettings('section_names', $defaultSectionNames);
    }

    /**
     * Save section names
     */
    public function saveSectionNames($sectionNames)
    {
        return $this->saveSettings('section_names', $sectionNames);
    }

    /**
     * Load section order
     */
    public function loadSectionOrder()
    {
        $config = include __DIR__ . '/../../config/config.php';
        $defaultOrder = $config['cv_defaults']['section_order'];
        
        return $this->loadSettings('section_order', $defaultOrder);
    }

    /**
     * Save section order
     */
    public function saveSectionOrder($sectionOrder)
    {
        return $this->saveSettings('section_order', $sectionOrder);
    }

    /**
     * Load template preference
     */
    public function loadTemplatePreference()
    {
        return $this->loadSettings('template_preference', 'modern');
    }

    /**
     * Save template preference
     */
    public function saveTemplatePreference($template)
    {
        return $this->saveSettings('template_preference', $template);
    }

    /**
     * Reset specific settings
     */
    public function resetSettings($settingsToReset)
    {
        $resetCount = 0;
        
        foreach ($settingsToReset as $setting) {
            switch ($setting) {
                case 'personal_info':
                    $this->deleteSetting('personal_info');
                    $resetCount++;
                    break;
                case 'experiences':
                    $this->deleteSetting('experiences');
                    $resetCount++;
                    break;
                case 'certifications':
                    $this->deleteSetting('certifications');
                    $resetCount++;
                    break;
                case 'languages':
                    $this->deleteSetting('languages');
                    $resetCount++;
                    break;
                case 'ai_settings':
                    $this->deleteSetting('ai_settings');
                    $resetCount++;
                    break;
                case 'section_names':
                    $this->deleteSetting('section_names');
                    $resetCount++;
                    break;
                case 'section_order':
                    $this->deleteSetting('section_order');
                    $resetCount++;
                    break;
                case 'template_preference':
                    $this->deleteSetting('template_preference');
                    $resetCount++;
                    break;
            }
        }
        
        return ['success' => true, 'reset_count' => $resetCount];
    }

    /**
     * Get settings file path for debugging
     */
    public function getSettingsPath()
    {
        return $this->settingsPath;
    }
}
